<?php

namespace App\Controllers;

use App\Libraries\CIAuth;

use App\Models\CIAuthModel;
use App\Models\PanelModel;
use App\Models\RadioModel;
use App\Models\CategoryModel;
use App\Models\CountriesModel;

class RadiosController extends BaseController {
    
    public function __construct(){
        $panelModel = new PanelModel();
        $this->settings = $panelModel->getSettings();
        $this->details = new RadioModel();
    }

    public function index() {
        if (!CIAuth::check()) {
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $data = [
            'pageTitle' => 'Manage Radios',
            'currentFile' => 'radios'
        ];
        
        $data['settings'] = $this->settings;
        
        // Get search keyword
        $keyword = $this->request->getGet('keyword') ?? '';
        
        if (!empty($keyword)) {
            // If searching, show all results without pagination
            $data['result'] = $this->details
                ->like('radio_title', $keyword)
                ->orderBy('id', 'DESC')
                ->findAll(); // No pagination here
                
            // Disable pagination variables
            $data['total_pages'] = 1; 
            $data['page'] = 1;
            $data['targetpage'] = '';
        } else {
            // Apply pagination only when not searching
            $limit = 12; // Number of items per page
            $page = $this->request->getGet('page') ?? 1;
            $page = max(1, intval($page));
            $offset = ($page - 1) * $limit;
            
            // Get total count only for non-search mode
            $total_records = $this->details->countAllResults();
            $total_pages = ($total_records > 0) ? ceil($total_records / $limit) : 1;
            
            $data['result'] = $this->details
                ->orderBy('id', 'DESC')
                ->findAll($limit, $offset);
                
            // Pass pagination data to view
            $data['total_pages'] = $total_pages;
            $data['page'] = $page;
            $data['targetpage'] = base_url('ns-admin/manage-radios');
        }
        return view('manage_radios', $data);
    }
    
    public function create(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $data = [
            'pageTitle' => 'Create Radio',
            'currentFile' => 'radios',
            'pageSave' => 'Create'
        ];
        
        $category = new CategoryModel();
        $Countries = new CountriesModel();
        
        $data['cat_result'] = $category->findAll();
        $data['cou_result'] = $Countries->findAll();
        
        $data['settings'] = $this->settings;
        return view('create_radio', $data);
    }
    
    public function edit($id = null){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $dataDetails = $this->details->find($id);
        
        $data = [
            'pageTitle' => 'Edit Radio',
            'currentFile' => 'radios',
            'pageSave' => 'Save',
            'radio_id' => $dataDetails['id'],
            'row' => $dataDetails,
        ];
        
        $category = new CategoryModel();
        $Countries = new CountriesModel();
        
        $data['cat_result'] = $category->findAll();
        $data['cou_result'] = $Countries->findAll();
        
        $data['settings'] = $this->settings;
        return view('create_radio', $data);
    }
    
    public function description($id = null){
        $dataDetails = $this->details->find($id);
        
        $data = [
            'pageTitle' => 'Description',
            'currentFile' => 'radios',
            'row' => $dataDetails,
        ];
        $data['settings'] = $this->settings;
        return view('policy', $data);
    }
    
    public function createHandler(){
        if(!CIAuth::check()){
            return redirect()->to(base_url('ns-admin/login'));
        }
        
        $editID = $this->request->getPost('radio_id');
        if(empty($editID)){
            
            $imageName = $this->handleImageUpload();
            $data = array( 
                'cat_id'  =>  $this->request->getPost('cat_id'),
                'countries_id'  =>  $this->request->getPost('countries_id'),
                'radio_title'  =>  $this->request->getPost('radio_title'),
                'isPremium'  =>   ($this->request->getPost('isPremium')) ? 'true' : 'false',
                'radio_url'  =>  $this->request->getPost('radio_url'),
                'radio_image'  =>  $imageName,
                'radio_description'  =>  addslashes(trim($this->request->getPost('radio_description'))),
                'status' => '1'
            );
            $this->details->insert($data);
            
            $message = array('message' => lang('Validation.added'),'class' => 'success');
            session()->set('response_msg', $message);
            return redirect()->to(base_url('ns-admin/create-radio'));
            
        } else {
            
            $dataDetails = $this->details->find($editID);
            if($dataDetails){
                if ($this->request->getPost('submit') !== null) {
                    $imageName = $this->handleImageUpload($dataDetails['radio_image']);
                    $data = array( 
                        'cat_id'  =>  $this->request->getPost('cat_id'),
                        'countries_id'  =>  $this->request->getPost('countries_id'),
                        'radio_title'  =>  $this->request->getPost('radio_title'),
                        'isPremium'  =>   ($this->request->getPost('isPremium')) ? 'true' : 'false',
                        'radio_url'  =>  $this->request->getPost('radio_url'),
                        'radio_image'  =>  $imageName,
                        'radio_description'  =>  addslashes(trim($this->request->getPost('radio_description')))
                    );
                    $this->details->update($editID, $data);
                    $message = array('message' => lang('Validation.updated'),'class' => 'success');
                } else {
                    $message = array('message' => lang('Validation.updated_failed'),'class' => 'error');
                }
            } else {
                $message = array('message' => lang('Validation.updated_failed_user'),'class' => 'error');
            }
            session()->set('response_msg', $message);
            return redirect()->to(base_url('ns-admin/create-radio/'.$editID));
        }
    }
    
    public function delete($id = null){
        // Check if the user is authorized to perform the action
        if (!CIAuth::check()) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Unauthorized']);
        }
        
        $allowedAdminTypes = [1, 3];
        if(!in_array(CIAuth::adminType(), $allowedAdminTypes)){
            return $this->response->setJSON(['status' => 0, 'message' => 'Unauthorized']);
        }
        
        // Validate ID
        if (is_null($id) || !is_numeric($id)) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Invalid ID']);
        }
        
        // Check if the record exists
        $record = $this->details->find($id);
        if (!$record) {
            return $this->response->setJSON(['status' => 0, 'message' => 'Record not found']);
        }
        
        try {
            // Attempt to delete the record
            if ($this->details->delete($id)) {
                
                $imageDirectory = 'images/';
                $existingImage = $record['radio_image'];
                if ($existingImage && file_exists($imageDirectory . $existingImage)) {
                    unlink($imageDirectory . $existingImage);
                }
                
                session()->set('response_msg', ['message' => lang('Validation.deleted'), 'class' => 'success']);
                return $this->response->setJSON(['status' => 1, 'message' => 'Delete successful']);
            } else {
                return $this->response->setJSON(['status' => 0, 'message' => 'Failed to delete']);
            }
        } catch (\Exception $e) {
            // Log and handle potential errors during deletion
            log_message('error', 'Error deleting record: ' . $e->getMessage());
            return $this->response->setJSON(['status' => 0, 'message' => 'An error occurred while deleting']);
        }
    }
    
    public function status($id = null){
        // Ensure user is authenticated
        if (!CIAuth::check()) {
            return $this->response->setStatusCode(401)
                ->setJSON(['status' => 0, 'msg' => 'Unauthorized']);
        }
        
        $allowedAdminTypes = [1, 3];
        if(!in_array(CIAuth::adminType(), $allowedAdminTypes)){
            return $this->response->setStatusCode(401)
                ->setJSON(['status' => 0, 'msg' => 'Unauthorized']);
        }
        
        // Validate ID
        if (is_null($id) || !is_numeric($id)) {
            return $this->response->setJSON(['status' => 0, 'msg' => 'Invalid ID']);
        }
        
        // Check if record exists
        $record = $this->details->find($id);
        if (!$record) {
            return $this->response->setJSON(['status' => 0, 'msg' => 'Record not found']);
        }
        
        // Ensure the request is a POST request
        if (!$this->request->is('post')) {
            return $this->response->setStatusCode(405)
                ->setJSON(['status' => 0, 'msg' => 'Method Not Allowed']);
        }
        
        try {
            $for_action = $this->request->getPost('for_action');
            $column = $this->request->getPost('column');
            
            // Update data
            $data = [$column => ($for_action === 'enable' ? '1' : '0')];
            $message = ($for_action === 'enable') ? lang('Validation.enabled') : lang('Validation.disabled');
            
            $this->details->update($id, $data);
            
            return $this->response->setJSON([
                'status' => 1,
                'action' => $for_action,
                'msg' => $message,
                'class' => 'success'
            ]);
        } catch (\Exception $e) {
            return $this->response->setJSON([
                'status' => 0,
                'msg' => 'An error occurred while updating the record'
            ]);
        }
    }
    
    private function handleImageUpload($existingImage = '') {
        $suffix = '_category';
        $file = $this->request->getFile('image_add');
        
        // Check if the file exists
        if (!$file) {
            return $existingImage; // Return the existing image if no file was uploaded
        }
        
        if ($file->isValid() && !$file->hasMoved()) {
            
            $imageDirectory = 'images/';
            
            // Check if the directory exists; if not, create it
            if (!is_dir($imageDirectory)) {
                mkdir($imageDirectory, 0777, true); // Create directory with proper permissions
            }
            
            // Delete existing image if it exists
            if ($existingImage && file_exists($imageDirectory . $existingImage)) {
                unlink($imageDirectory . $existingImage);
            }
            
            // Generate new image name with a suffix
            $extension = $file->getExtension(); // Get file extension
            $imageName = rand(0, 99999) . $suffix . '.' . $extension; // Append suffix
            
            // Move the file with the new name
            $file->move($imageDirectory, $imageName);
            
            return $imageName;
        }
        return $existingImage;
    }
}
